/*
 * Decompiled with CFR 0.152.
 */
package io.grpc.internal;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.MoreObjects;
import com.google.common.base.Objects;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.base.Verify;
import io.grpc.CallOptions;
import io.grpc.Channel;
import io.grpc.ClientCall;
import io.grpc.ClientInterceptor;
import io.grpc.Deadline;
import io.grpc.MethodDescriptor;
import io.grpc.Status;
import io.grpc.internal.HedgingPolicy;
import io.grpc.internal.RetryPolicy;
import io.grpc.internal.ServiceConfigUtil;
import java.util.Collections;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.annotation.CheckForNull;
import javax.annotation.Nonnull;

final class ServiceConfigInterceptor
implements ClientInterceptor {
    private static final Logger logger = Logger.getLogger(ServiceConfigInterceptor.class.getName());
    @VisibleForTesting
    final AtomicReference<Map<String, MethodInfo>> serviceMethodMap = new AtomicReference();
    @VisibleForTesting
    final AtomicReference<Map<String, MethodInfo>> serviceMap = new AtomicReference();
    private final boolean retryEnabled;
    private final int maxRetryAttemptsLimit;
    private final int maxHedgedAttemptsLimit;
    private volatile boolean nameResolveComplete;
    static final CallOptions.Key<RetryPolicy.Provider> RETRY_POLICY_KEY = CallOptions.Key.create("internal-retry-policy");
    static final CallOptions.Key<HedgingPolicy.Provider> HEDGING_POLICY_KEY = CallOptions.Key.create("internal-hedging-policy");

    ServiceConfigInterceptor(boolean retryEnabled, int maxRetryAttemptsLimit, int maxHedgedAttemptsLimit) {
        this.retryEnabled = retryEnabled;
        this.maxRetryAttemptsLimit = maxRetryAttemptsLimit;
        this.maxHedgedAttemptsLimit = maxHedgedAttemptsLimit;
    }

    void handleUpdate(@Nonnull Map<String, Object> serviceConfig) {
        HashMap<String, MethodInfo> newServiceMethodConfigs = new HashMap<String, MethodInfo>();
        HashMap<String, MethodInfo> newServiceConfigs = new HashMap<String, MethodInfo>();
        List<Map<String, Object>> methodConfigs = ServiceConfigUtil.getMethodConfigFromServiceConfig(serviceConfig);
        if (methodConfigs == null) {
            logger.log(Level.FINE, "No method configs found, skipping");
            this.nameResolveComplete = true;
            return;
        }
        for (Map<String, Object> methodConfig : methodConfigs) {
            MethodInfo info = new MethodInfo(methodConfig, this.retryEnabled, this.maxRetryAttemptsLimit, this.maxHedgedAttemptsLimit);
            List<Map<String, Object>> nameList = ServiceConfigUtil.getNameListFromMethodConfig(methodConfig);
            Preconditions.checkArgument((nameList != null && !nameList.isEmpty() ? 1 : 0) != 0, (String)"no names in method config %s", methodConfig);
            for (Map<String, Object> name : nameList) {
                String serviceName = ServiceConfigUtil.getServiceFromName(name);
                Preconditions.checkArgument((!Strings.isNullOrEmpty((String)serviceName) ? 1 : 0) != 0, (Object)"missing service name");
                String methodName = ServiceConfigUtil.getMethodFromName(name);
                if (Strings.isNullOrEmpty((String)methodName)) {
                    Preconditions.checkArgument((!newServiceConfigs.containsKey(serviceName) ? 1 : 0) != 0, (String)"Duplicate service %s", (Object)serviceName);
                    newServiceConfigs.put(serviceName, info);
                    continue;
                }
                String fullMethodName = MethodDescriptor.generateFullMethodName(serviceName, methodName);
                Preconditions.checkArgument((!newServiceMethodConfigs.containsKey(fullMethodName) ? 1 : 0) != 0, (String)"Duplicate method name %s", (Object)fullMethodName);
                newServiceMethodConfigs.put(fullMethodName, info);
            }
        }
        this.serviceMethodMap.set(Collections.unmodifiableMap(newServiceMethodConfigs));
        this.serviceMap.set(Collections.unmodifiableMap(newServiceConfigs));
        this.nameResolveComplete = true;
    }

    private static HedgingPolicy hedgingPolicy(Map<String, Object> hedgingPolicy, int maxAttemptsLimit) {
        int maxAttempts = (Integer)Preconditions.checkNotNull((Object)ServiceConfigUtil.getMaxAttemptsFromHedgingPolicy(hedgingPolicy), (Object)"maxAttempts cannot be empty");
        Preconditions.checkArgument((maxAttempts >= 2 ? 1 : 0) != 0, (String)"maxAttempts must be greater than 1: %s", (int)maxAttempts);
        maxAttempts = Math.min(maxAttempts, maxAttemptsLimit);
        long hedgingDelayNanos = (Long)Preconditions.checkNotNull((Object)ServiceConfigUtil.getHedgingDelayNanosFromHedgingPolicy(hedgingPolicy), (Object)"hedgingDelay cannot be empty");
        Preconditions.checkArgument((hedgingDelayNanos >= 0L ? 1 : 0) != 0, (String)"hedgingDelay must not be negative: %s", (long)hedgingDelayNanos);
        List<String> rawCodes = ServiceConfigUtil.getNonFatalStatusCodesFromHedgingPolicy(hedgingPolicy);
        Preconditions.checkNotNull(rawCodes, (Object)"rawCodes must be present");
        Preconditions.checkArgument((!rawCodes.isEmpty() ? 1 : 0) != 0, (Object)"rawCodes can't be empty");
        EnumSet<Status.Code> codes = EnumSet.noneOf(Status.Code.class);
        for (String rawCode : rawCodes) {
            Verify.verify((!"OK".equals(rawCode) ? 1 : 0) != 0, (String)"rawCode can not be \"OK\"", (Object[])new Object[0]);
            codes.add(Status.Code.valueOf(rawCode));
        }
        Set<Status.Code> nonFatalStatusCodes = Collections.unmodifiableSet(codes);
        return new HedgingPolicy(maxAttempts, hedgingDelayNanos, nonFatalStatusCodes);
    }

    @Override
    public <ReqT, RespT> ClientCall<ReqT, RespT> interceptCall(final MethodDescriptor<ReqT, RespT> method, CallOptions callOptions, Channel next) {
        Integer existingLimit;
        MethodInfo info;
        if (this.retryEnabled) {
            if (this.nameResolveComplete) {
                final RetryPolicy retryPolicy = this.getRetryPolicyFromConfig(method);
                final HedgingPolicy hedgingPolicy = this.getHedgingPolicyFromConfig(method);
                Verify.verify((retryPolicy.equals(RetryPolicy.DEFAULT) || hedgingPolicy.equals(HedgingPolicy.DEFAULT) ? 1 : 0) != 0, (String)"Can not apply both retry and hedging policy for the method '%s'", (Object[])new Object[]{method});
                final class ImmediateRetryPolicyProvider
                implements RetryPolicy.Provider {
                    ImmediateRetryPolicyProvider() {
                    }

                    @Override
                    public RetryPolicy get() {
                        return retryPolicy;
                    }
                }
                final class ImmediateHedgingPolicyProvider
                implements HedgingPolicy.Provider {
                    ImmediateHedgingPolicyProvider() {
                    }

                    @Override
                    public HedgingPolicy get() {
                        return hedgingPolicy;
                    }
                }
                callOptions = callOptions.withOption(RETRY_POLICY_KEY, new ImmediateRetryPolicyProvider()).withOption(HEDGING_POLICY_KEY, new ImmediateHedgingPolicyProvider());
            } else {
                final class DelayedRetryPolicyProvider
                implements RetryPolicy.Provider {
                    DelayedRetryPolicyProvider() {
                    }

                    @Override
                    public RetryPolicy get() {
                        if (!ServiceConfigInterceptor.this.nameResolveComplete) {
                            return RetryPolicy.DEFAULT;
                        }
                        return ServiceConfigInterceptor.this.getRetryPolicyFromConfig(method);
                    }
                }
                final class DelayedHedgingPolicyProvider
                implements HedgingPolicy.Provider {
                    DelayedHedgingPolicyProvider() {
                    }

                    @Override
                    public HedgingPolicy get() {
                        if (!ServiceConfigInterceptor.this.nameResolveComplete) {
                            return HedgingPolicy.DEFAULT;
                        }
                        HedgingPolicy hedgingPolicy = ServiceConfigInterceptor.this.getHedgingPolicyFromConfig(method);
                        Verify.verify((hedgingPolicy.equals(HedgingPolicy.DEFAULT) || ServiceConfigInterceptor.this.getRetryPolicyFromConfig(method).equals(RetryPolicy.DEFAULT) ? 1 : 0) != 0, (String)"Can not apply both retry and hedging policy for the method '%s'", (Object[])new Object[]{method});
                        return hedgingPolicy;
                    }
                }
                callOptions = callOptions.withOption(RETRY_POLICY_KEY, new DelayedRetryPolicyProvider()).withOption(HEDGING_POLICY_KEY, new DelayedHedgingPolicyProvider());
            }
        }
        if ((info = this.getMethodInfo(method)) == null) {
            return next.newCall(method, callOptions);
        }
        if (info.timeoutNanos != null) {
            Deadline newDeadline = Deadline.after((long)info.timeoutNanos, (TimeUnit)TimeUnit.NANOSECONDS);
            Deadline existingDeadline = callOptions.getDeadline();
            if (existingDeadline == null || newDeadline.compareTo(existingDeadline) < 0) {
                callOptions = callOptions.withDeadline(newDeadline);
            }
        }
        if (info.waitForReady != null) {
            CallOptions callOptions2 = callOptions = info.waitForReady != false ? callOptions.withWaitForReady() : callOptions.withoutWaitForReady();
        }
        if (info.maxInboundMessageSize != null) {
            existingLimit = callOptions.getMaxInboundMessageSize();
            callOptions = existingLimit != null ? callOptions.withMaxInboundMessageSize(Math.min(existingLimit, info.maxInboundMessageSize)) : callOptions.withMaxInboundMessageSize(info.maxInboundMessageSize);
        }
        if (info.maxOutboundMessageSize != null) {
            existingLimit = callOptions.getMaxOutboundMessageSize();
            callOptions = existingLimit != null ? callOptions.withMaxOutboundMessageSize(Math.min(existingLimit, info.maxOutboundMessageSize)) : callOptions.withMaxOutboundMessageSize(info.maxOutboundMessageSize);
        }
        return next.newCall(method, callOptions);
    }

    @CheckForNull
    private MethodInfo getMethodInfo(MethodDescriptor<?, ?> method) {
        Map<String, MethodInfo> localServiceMap;
        Map<String, MethodInfo> localServiceMethodMap = this.serviceMethodMap.get();
        MethodInfo info = null;
        if (localServiceMethodMap != null) {
            info = localServiceMethodMap.get(method.getFullMethodName());
        }
        if (info == null && (localServiceMap = this.serviceMap.get()) != null) {
            info = localServiceMap.get(MethodDescriptor.extractFullServiceName(method.getFullMethodName()));
        }
        return info;
    }

    @VisibleForTesting
    RetryPolicy getRetryPolicyFromConfig(MethodDescriptor<?, ?> method) {
        MethodInfo info = this.getMethodInfo(method);
        return info == null ? RetryPolicy.DEFAULT : info.retryPolicy;
    }

    @VisibleForTesting
    HedgingPolicy getHedgingPolicyFromConfig(MethodDescriptor<?, ?> method) {
        MethodInfo info = this.getMethodInfo(method);
        return info == null ? HedgingPolicy.DEFAULT : info.hedgingPolicy;
    }

    static final class MethodInfo {
        final Long timeoutNanos;
        final Boolean waitForReady;
        final Integer maxInboundMessageSize;
        final Integer maxOutboundMessageSize;
        final RetryPolicy retryPolicy;
        final HedgingPolicy hedgingPolicy;

        MethodInfo(Map<String, Object> methodConfig, boolean retryEnabled, int maxRetryAttemptsLimit, int maxHedgedAttemptsLimit) {
            this.timeoutNanos = ServiceConfigUtil.getTimeoutFromMethodConfig(methodConfig);
            this.waitForReady = ServiceConfigUtil.getWaitForReadyFromMethodConfig(methodConfig);
            this.maxInboundMessageSize = ServiceConfigUtil.getMaxResponseMessageBytesFromMethodConfig(methodConfig);
            if (this.maxInboundMessageSize != null) {
                Preconditions.checkArgument((this.maxInboundMessageSize >= 0 ? 1 : 0) != 0, (String)"maxInboundMessageSize %s exceeds bounds", (Object)this.maxInboundMessageSize);
            }
            this.maxOutboundMessageSize = ServiceConfigUtil.getMaxRequestMessageBytesFromMethodConfig(methodConfig);
            if (this.maxOutboundMessageSize != null) {
                Preconditions.checkArgument((this.maxOutboundMessageSize >= 0 ? 1 : 0) != 0, (String)"maxOutboundMessageSize %s exceeds bounds", (Object)this.maxOutboundMessageSize);
            }
            Map<String, Object> retryPolicyMap = retryEnabled ? ServiceConfigUtil.getRetryPolicyFromMethodConfig(methodConfig) : null;
            this.retryPolicy = retryPolicyMap == null ? RetryPolicy.DEFAULT : MethodInfo.retryPolicy(retryPolicyMap, maxRetryAttemptsLimit);
            Map<String, Object> hedgingPolicyMap = retryEnabled ? ServiceConfigUtil.getHedgingPolicyFromMethodConfig(methodConfig) : null;
            this.hedgingPolicy = hedgingPolicyMap == null ? HedgingPolicy.DEFAULT : ServiceConfigInterceptor.hedgingPolicy(hedgingPolicyMap, maxHedgedAttemptsLimit);
        }

        public int hashCode() {
            return Objects.hashCode((Object[])new Object[]{this.timeoutNanos, this.waitForReady, this.maxInboundMessageSize, this.maxOutboundMessageSize, this.retryPolicy});
        }

        public boolean equals(Object other) {
            if (!(other instanceof MethodInfo)) {
                return false;
            }
            MethodInfo that = (MethodInfo)other;
            return Objects.equal((Object)this.timeoutNanos, (Object)that.timeoutNanos) && Objects.equal((Object)this.waitForReady, (Object)that.waitForReady) && Objects.equal((Object)this.maxInboundMessageSize, (Object)that.maxInboundMessageSize) && Objects.equal((Object)this.maxOutboundMessageSize, (Object)that.maxOutboundMessageSize) && Objects.equal((Object)this.retryPolicy, (Object)that.retryPolicy);
        }

        public String toString() {
            return MoreObjects.toStringHelper((Object)this).add("timeoutNanos", (Object)this.timeoutNanos).add("waitForReady", (Object)this.waitForReady).add("maxInboundMessageSize", (Object)this.maxInboundMessageSize).add("maxOutboundMessageSize", (Object)this.maxOutboundMessageSize).add("retryPolicy", (Object)this.retryPolicy).toString();
        }

        private static RetryPolicy retryPolicy(Map<String, Object> retryPolicy, int maxAttemptsLimit) {
            int maxAttempts = (Integer)Preconditions.checkNotNull((Object)ServiceConfigUtil.getMaxAttemptsFromRetryPolicy(retryPolicy), (Object)"maxAttempts cannot be empty");
            Preconditions.checkArgument((maxAttempts >= 2 ? 1 : 0) != 0, (String)"maxAttempts must be greater than 1: %s", (int)maxAttempts);
            maxAttempts = Math.min(maxAttempts, maxAttemptsLimit);
            long initialBackoffNanos = (Long)Preconditions.checkNotNull((Object)ServiceConfigUtil.getInitialBackoffNanosFromRetryPolicy(retryPolicy), (Object)"initialBackoff cannot be empty");
            Preconditions.checkArgument((initialBackoffNanos > 0L ? 1 : 0) != 0, (String)"initialBackoffNanos must be greater than 0: %s", (long)initialBackoffNanos);
            long maxBackoffNanos = (Long)Preconditions.checkNotNull((Object)ServiceConfigUtil.getMaxBackoffNanosFromRetryPolicy(retryPolicy), (Object)"maxBackoff cannot be empty");
            Preconditions.checkArgument((maxBackoffNanos > 0L ? 1 : 0) != 0, (String)"maxBackoff must be greater than 0: %s", (long)maxBackoffNanos);
            double backoffMultiplier = (Double)Preconditions.checkNotNull((Object)ServiceConfigUtil.getBackoffMultiplierFromRetryPolicy(retryPolicy), (Object)"backoffMultiplier cannot be empty");
            Preconditions.checkArgument((backoffMultiplier > 0.0 ? 1 : 0) != 0, (String)"backoffMultiplier must be greater than 0: %s", (Object)backoffMultiplier);
            List<String> rawCodes = ServiceConfigUtil.getRetryableStatusCodesFromRetryPolicy(retryPolicy);
            Preconditions.checkNotNull(rawCodes, (Object)"rawCodes must be present");
            Preconditions.checkArgument((!rawCodes.isEmpty() ? 1 : 0) != 0, (Object)"rawCodes can't be empty");
            EnumSet<Status.Code> codes = EnumSet.noneOf(Status.Code.class);
            for (String rawCode : rawCodes) {
                Verify.verify((!"OK".equals(rawCode) ? 1 : 0) != 0, (String)"rawCode can not be \"OK\"", (Object[])new Object[0]);
                codes.add(Status.Code.valueOf(rawCode));
            }
            Set<Status.Code> retryableStatusCodes = Collections.unmodifiableSet(codes);
            return new RetryPolicy(maxAttempts, initialBackoffNanos, maxBackoffNanos, backoffMultiplier, retryableStatusCodes);
        }
    }
}

