package com.bruce.tool.netty.socket.core;

import com.bruce.tool.common.util.LogUtils;
import com.bruce.tool.common.util.string.JsonUtils;
import com.bruce.tool.common.util.string.MapHandler;
import com.bruce.tool.common.util.string.StringUtils;
import com.bruce.tool.netty.socket.util.SocketUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.Unpooled;
import io.netty.channel.Channel;
import io.netty.channel.ChannelHandler.Sharable;
import io.netty.channel.ChannelHandlerContext;
import io.netty.channel.ChannelInboundHandlerAdapter;
import lombok.extern.slf4j.Slf4j;
import org.springframework.util.CollectionUtils;

import java.util.List;
import java.util.Map;
import java.util.Objects;

@Slf4j
@Sharable
public class SocketServerHandler extends ChannelInboundHandlerAdapter {

    private SocketServer server;

    public SocketServerHandler(SocketServer server) {
        this.server = server;
    }

    private Map<String,Channel> clients = Maps.newConcurrentMap();
    private List<Channel> servers = Lists.newArrayList();

    @Override
    public void channelRead(ChannelHandlerContext ctx, Object msg) {
        if( !( msg instanceof ByteBuf ) ){ return; }
        String message = SocketUtils.transfer((ByteBuf) msg);
        if (StringUtils.isEmpty(message)){
            return;
        }
        Channel incoming = ctx.channel();
        // 发送心跳检测
        if( message.contains(server.getConfig().getPing()) ){
            LogUtils.info(log,"心跳检测:{}",message);
            Map ping = JsonUtils.strToObj(message,Map.class);
            String clientKey = (String) ping.get(server.getConfig().getPing());
            if(StringUtils.isNotBlank(clientKey)){
                if(clientKey.startsWith("server")){
                    servers.add(incoming);
                }else{
                    clients.put(clientKey,incoming);
                }
            }
            Map body = MapHandler.build().add(server.getConfig().getPong(),"服务器回复心跳");
            message = JsonUtils.objToStr(body);
            msg = Unpooled.copiedBuffer(message.getBytes());
            incoming.writeAndFlush(msg);
            return;
        }
        //发送业务数据
        Map data = JsonUtils.strToObj(message,Map.class);
        if(CollectionUtils.isEmpty(data)){
            LogUtils.debug(log,"错误的消息格式");
            return;
        }
        String clientKey = (String) data.get(server.getConfig().getKey());
        if(StringUtils.isBlank(clientKey)){
            LogUtils.debug(log,"错误的消息格式,缺少客户端Key");
            return;
        }
        Object isServer = data.get("isServer");
        if( Objects.nonNull(isServer) ){ // 消息来源于客户端中的服务者,消息需要发送给客户
            Channel channel = clients.get(clientKey);
            if(Objects.nonNull(channel) ){
                channel.writeAndFlush(msg);
            }
        }else{ //消息来源于客户端中的客户,消息需要发送给服务者
            Integer size = servers.size();
            Channel channel = servers.get((int)(Math.random()*size));
            if(Objects.nonNull(channel) ){
                channel.writeAndFlush(msg);
            }
        }
    }

}