package com.bruce.tool.netty.socket.core;

import com.bruce.tool.common.exception.BaseRuntimeException;
import com.bruce.tool.common.util.LogUtils;
import com.bruce.tool.common.util.string.JsonUtils;
import com.bruce.tool.common.util.string.MapHandler;
import com.bruce.tool.netty.socket.constant.NettyErrorCode;
import com.bruce.tool.netty.socket.util.SocketUtils;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.Unpooled;
import io.netty.channel.ChannelHandler.Sharable;
import io.netty.channel.ChannelHandlerContext;
import io.netty.channel.ChannelInboundHandlerAdapter;
import io.netty.handler.timeout.IdleStateEvent;
import lombok.extern.slf4j.Slf4j;

import java.util.Map;
import java.util.Objects;

/**
 * Keep reconnecting to the server while printing out the current uptime and
 * connection attempt getStatus.
 */
@Slf4j
@Sharable
public class SocketClientHandler extends ChannelInboundHandlerAdapter{

    private SocketClient client;

    public SocketClientHandler(SocketClient client) {
        this.client = client;
    }

    @Override
    public void channelRead(ChannelHandlerContext ctx, Object msg) {
        if( !( msg instanceof ByteBuf ) ){ return; }
        String message = SocketUtils.transfer((ByteBuf) msg);
        if(message.contains(client.config().getPong())){
            LogUtils.info(log,"心跳检测:{}",message);
        }else{
            if(Objects.isNull(client.getListener()) ){
                throw new BaseRuntimeException(NettyErrorCode.NTERROR_CONFIG_NULL.getCode(),"监听器未初始化");
            }
            client.getListener().listen(message);
        }
    }

    @Override
    public void channelInactive(final ChannelHandlerContext ctx) {
        client.connect();
    }

    @Override
    public void userEventTriggered(ChannelHandlerContext ctx, Object evt) {
        if (!(evt instanceof IdleStateEvent)) {
            return;
        }

        IdleStateEvent e = (IdleStateEvent) evt;
        switch(e.state()){
            case READER_IDLE:
                handleReaderIdle(ctx);
                break;
            case WRITER_IDLE:
                handleWriterIdle(ctx);
                break;
            case ALL_IDLE:
                handleAllIdle(ctx);
                break;
            default:
                break;
        }
    }

    /**状态为读闲置**/
    private void handleReaderIdle(ChannelHandlerContext ctx) {
        ctx.close();
    }

    /**状态为写闲置**/
    private void handleWriterIdle(ChannelHandlerContext ctx) {
        // do nothing
    }

    /**读写均闲置**/
    private void handleAllIdle(ChannelHandlerContext ctx) {
        Map body = MapHandler.build().add(client.config().getPing(),client.config().getUniqueNo());
        String message = JsonUtils.objToStr(body);
        ctx.writeAndFlush(Unpooled.copiedBuffer(message.getBytes()));
    }

}