package com.bruce.tool.common.util.regex;

import com.google.common.collect.Lists;
import lombok.AccessLevel;
import lombok.NoArgsConstructor;
import org.apache.commons.lang3.StringUtils;

import java.net.URLDecoder;
import java.net.URLEncoder;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * 功能:
 * 正则表达式
 * @author : wuyujia 2017/10/31 12:06
 * @author : bruce 2018/05/26 12:46
 */
@NoArgsConstructor(access = AccessLevel.PRIVATE)
public class RegexUtils {

    /**
     * 非法字符正则校验
     */
    private static final Pattern LEGAL_PATTERN = Pattern.compile("^[\\u4e00-\\u9fa5\\w\\d_.%@!！\\[\\]\\\\|\\-,/,\\[\\]\\s，。;；:：“”?？]+$");

    /**
     * 纯数字校验
     */
    private static final Pattern NUMBER_PATTERN = Pattern.compile("^\\d+$");

    /**
     * 用户名正则表达式
     */
    private static final Pattern USER_NAME_PATTERN = Pattern.compile("^[a-zA-Z0-9]{4,20}$");

    /**
     * 手机号正则表达式
     */
    private static final Pattern PHONE_NUMBER_PATTERN = Pattern.compile("^1[345789]\\d{9}$");

    /**
     * 字母数字正则表达式
     */
    private static final Pattern ALPHA_NUM = Pattern.compile("^[a-zA-Z0-9]+$");

    /**
     * 空串校验
     */
    private static final Pattern EMPTY = Pattern.compile("^\\s*$");

    /**
     * 是否包含中文的校验
     * @param s 需要校验的字符串
     * @return true 包含, false 不包含
     */
    public static boolean containChinese(String s) {
        if (s != null && !s.isEmpty()) {
            Matcher matcher = Pattern.compile(Regex.CHINESE.getValue()).matcher(s);
            if (matcher.find()) {
                return true;
            }
        }
        return false;
    }

    /**
     * 判断是否包含中文, 且长度小于length
     * @param s 需要校验的字符串
     * @param length 需要判断的长度
     * @return true 包含中文, 且长度小于length, false 不包含中文, 或长度大于length
     */
    public static boolean containChinese(String s, int length) {
        if (containChinese(s)) {
            return s.length() <= length;
        }
        return false;
    }

    /**
     * 字符串长度限制
     * @param s 待检验的字符串
     * @param length 限制长度 <=
     * @return false 未超出, true 超出
     */
    public static boolean overLengthLimit(String s, int length) {
        if (s != null) {
            return s.length() > length;
        }
        return false;
    }

    /**
     * 不包含中文
     * @param s 需要校验的字符串
     * @return true 不包含, false 包含
     */
    public static boolean notContainChinese(String s) {
        return !containChinese(s);
    }

    /**
     * 不包含非法字符校验 (除 中文, 英文, 数字, _ 以外的都属于非法字符)
     * @param s 需要校验的字符串
     * @return false 包含非法字符,
     */
    public static boolean notContainIllegalParameter(String s) {
        if (s == null || s.isEmpty()) {
            // 如果为空, 判断不包含非法字符
            return true;
        }
        Matcher matcher = LEGAL_PATTERN.matcher(s);
        return matcher.matches();
    }

    /**
     * 只有整数的校验方法
     * @param s 待校验的字符串
     * @return
     * true 只有数字
     * false 不含有数字, 为空, 或者含有其他字符
     */
    public static boolean onlyIntegerNumber(String s) {
        if (s == null || s.isEmpty()) {
            return false;
        }
        Matcher matcher = NUMBER_PATTERN.matcher(s);
        return matcher.matches();
    }

    /**
     * 手机号校验
     * @param s
     * @return
     * true 是合法手机号
     * false 不是手机号或者为空
     */
    public static boolean isPhoneNumber(String s) {
        if (s == null || s.isEmpty()) {
            return false;
        }
        Matcher matcher = PHONE_NUMBER_PATTERN.matcher(s);
        return matcher.matches();
    }

    public static boolean isUserName(String un){
        if(StringUtils.isBlank(un)){
            return false;
        }
        Matcher matcher = USER_NAME_PATTERN.matcher(un);
        return matcher.matches();
    }

    /**
     * 仅仅包含字母数字的校验
     * @param s
     */
    public static boolean onlyAlphaNum (String s) {
        if (StringUtils.isBlank(s)) {
            return false;
        }
        Matcher matcher = ALPHA_NUM.matcher(s);
        return matcher.matches();
    }


    /**
     * 空判断
     * @param s
     * @return
     */
    public static boolean isEmpty(String s) {
        if (null == s) {
            return true;
        }
        if (EMPTY.matcher(s).matches()) {
            return true;
        }
        return false;
    }

    public static String urlDecode(String param){
        if (StringUtils.isNotBlank(param)){
            String result = URLDecoder.decode(param);
            if( StringUtils.isNotBlank(result) ){
                return result;
            }
        }
        return param;
    }

    public static String urlEncode(String param){
        if (StringUtils.isNotBlank(param)){
            String result = URLEncoder.encode(param);
            if( StringUtils.isNotBlank(result) ){
                return result;
            }
        }
        return param;
    }

    public static String urlEncodeOneByOne(String param){
        String[] list = param.split("");
        StringBuilder result = new StringBuilder();
        for ( String item : list ) {
            if( containChinese(item) ){
                result.append(urlEncode(item));
            }else{
                result.append(item);
            }
        }
        return result.toString();
    }

    public static boolean containsOne(String string,String regex){
        Pattern p1 = Pattern.compile(regex);
        Matcher m1 = p1.matcher(string);
        Integer count = 0;
        while(m1.find()) {
            count ++ ;
            if( count > 1 ){
                return false;
            }
        }
        return true;
    }

    public static boolean containsMore(String string,String regex,Integer time){
        if( time <= 0 ){ return false; }
        Pattern p1 = Pattern.compile(regex);
        Matcher m1 = p1.matcher(string);
        Integer count = 0;
        do {
            count ++ ;
            if( count > time ){
                return true;
            }
        }while(m1.find());
        return false;
    }

    /**
     * 统计字符在文本中出现的次数
     **/
    public static List<String> gatherStringCount(String content,String regex) {
        Pattern p1 = Pattern.compile(regex);
        Matcher m1 = p1.matcher(content);
        List<String> colgroups = Lists.newArrayList();
        while (m1.find()) {
            for (int i = 0; i < m1.groupCount(); i++) {
                String test1 = m1.group(i);
                colgroups.add(test1);
            }
        }
        return colgroups;
    }

}
